# -*- coding: utf-8 -*-

from functools import wraps
from threading import Thread
import socket
import tkinter as tk

def singleton(o_cls):
    orig_new = o_cls.__new__
    inst_field = "__instance"

    @wraps(o_cls.__new__)
    def __new__(cls, *args, **kwargs):
        if (instance := getattr(cls, inst_field, None)) is None:
            instance = orig_new(cls)
            if hasattr(cls, "__init__"):
                cls.__init__(instance, *args, **kwargs)
                delattr(cls, "__init__")
            setattr(cls, inst_field, instance)
        return instance

    o_cls.__new__ = __new__
    return o_cls


@singleton
class Delivery:
    ''' Un "postino" tra moduli'''
    
    def __init__(self):
        self.subscr = {}
    
    def subscribe(self, group: str, method: any) -> None:
        ''' Aggiunge un subscriber ad un dato gruppo '''
        if not group in self.subscr.keys():
            self.subscr[group] = []
        if method in self.subscr[group]:
            raise RuntimeError('Metodo già acquisito nel gruppo %s' % group)
        self.subscr[group].append(method)

    def unsubscribe(self, group: str, method: any) -> None:
        ''' Rimuove un subscriber ad un dato gruppo, i gruppi vuoti vengono rimossi'''
        if not group in self.subscr.keys():
            raise RuntimeError('Gruppo %s non registrato' % group)
        if method not in self.subscr[group]:
            raise RuntimeError('Metodo non registrato nel gruppo %s' % group)
        self.subscr[group].remove(method)
        if not self.subscr[group]:
            del self.subscr[group]
    
    def send_message(self, group:str, message: list) -> None:
        ''' Distribuisce un messaggio ai subscriber di un gruppo '''
        if not group in self.subscr.keys():
            raise RuntimeError('Gruppo %s non registrato' % group)
        for s_func in self.subscr[group]:
            s_func(message)
            

class Chatterbox:
    def __init__(self, svr: str, port: int) -> None:
        self.svr = svr
        self.port = port
        self.sok = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self.pub = Delivery()
        self.pub.subscribe('WORKER', self.obs)
        self.still = True

    def connect(self) -> None:
        try:
            self.sok.connect((self.svr, self.port))
            self.sok.settimeout(0.5)
            msg = ['OUTCOME', f' Connessione con {self.svr} stabilita']
            self.pub.send_message('OBSERVER', msg)
            # lancia un thread per l'ascolto
            t = Thread(target=self.listen)
            t.daemon = True
            t.start()
        except OSError as e:
            msg = ['ERROR', repr(e)]
            self.pub.send_message('OBSERVER', msg)

    def listen(self) -> None:
        while self.still:
            try:
                data = self.sok.recv(2048)
                #print(data)
                # qui andrebbe una queue per collezionare i dati giunti
                # ma complicherebbe le cose, mi limito a comunicare
                msg = ['RECEIVED', data[87:-3].decode(encoding="utf-8", errors="ignore")]
                self.pub.send_message('LISTENERS', msg)
            except socket.timeout:
                continue
        self.sok.close()
        msg = ['OUTCOME', f'Connessione con {self.svr} chiusa']
        self.pub.send_message('OBSERVER', msg)
        self.pub.unsubscribe('WORKER', self.obs)

    def transmit(self, message: bytes) -> None:
        try:
            self.sok.sendall(message)
        except OSError as e:
            msg = ['ERROR', repr(e)]
            self.pub.send_message('OBSERVER', msg)
    
    def close(self) -> None:
        self.still = False

    def obs(self, message):
        op = message[0]
        if op == 'CLOSING' and message[0] == 'CLOSING':
            self.still = False
        
class BufferedText(tk.Text):
    '''Una Text-box inibibile alla scrittura senza disabilitazione
       e che espone un numero stabilito di righe di testo.'''
    key_none = ['Alt_L', 'Alt_R', 'Caps_Lock', 'Control_L', 'Control_R', 'Down', 'End',
                'Escape', 'Execute', 'F1', 'F2', 'Fi', 'F12', 'Home', 'Insert', 'Left',
                'Linefeed', 'KP_Add', 'KP_Begin', 'KP_Divide', 'KP_Down', 'KP_End',
                'KP_Home', 'KP_Insert', 'KP_Left', 'KP_Next', 'KP_Prior','KP_Right',
                'KP_Up', 'Next', 'Num_Lock', 'Pause', 'Print', 'Prior', 'Right',
                'Scroll_Lock', 'Shift_L', 'Shift_R', 'Tab', 'Up']
    def __init__(self, parent: callable, active: bool=False, rowsize: int=300, *args, **kwargs) -> None:
        super().__init__(parent, *args, **kwargs)
        self.parent = parent
        self._active = active
        self._rowsize =  rowsize if rowsize >=0 else 0
        self._bg = self['background']
        self.bind('<KeyPress>', self._on_key)
        self.bind('<FocusIn>', self._on_focus)
        self.bind('<FocusOut>', self._out_focus)

    def _on_key(self, evt: callable) -> None:
        if  evt.keysym in ['Return', 'KP_Enter'] and self._rowsize:  # valuta il caso si aggiunga una riga
            if not self._active: return 'break'
            self._evaluate_rows()
        elif not self._active and not evt.keysym in self.key_none:
            return 'break'

    def _evaluate_rows(self):
        if not self._rowsize: return
        rows = int(self.index('end').split('.')[0]) - 2
        if self._active and self.parent.focus_get() == self:
            rows += 1
        if rows > self._rowsize:  # raggiunto il limite di righe stabilito
            for i in range(rows - self._rowsize):
                self.delete('1.0', '1.end + 1 char')
                self.update()

    def _on_focus(self, evt: callable) -> None:
        color = '#ffffc0' if self._active else '#bfe5f1'
        self.configure(bg=color)

    def _out_focus(self, evt: callable) -> None:
        self.configure(bg=self._bg)

    def is_active(self) -> bool:
        return self._active

    def activate(self) -> None:
        self._active = True

    def disable(self) -> None:
        self._active = False

    @property
    def buffer(self) -> int:
        return self._rowsize

    @buffer.setter
    def buffer(self, buff: int) -> None:
        self._rowsize = buff if buff >=0 else self._rowsize
        self._evaluate_rows()

    def add_text(self, text: str) -> None:
        self.insert('end', text)
        self._evaluate_rows()
        self.see('end')
